<?php

namespace Modules\Artist\Http\Controllers;

use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Modules\Artist\Entities\Artist;
use Modules\Artist\Entities\ArtistGenre;
use DataTables;
use Illuminate\Support\Carbon;
use Str;
use Illuminate\Support\Facades\Validator;

class ArtistController extends Controller
{
    public function index(){
        return view('artist::artist.index');
    }

    public function artistData(){
        $artists = select(['column' => ['artists.id','artists.image','artists.artist_name','artists.dob', 'artist_genres.genre_name', 'artists.status', 'artists.created_at'], 'table' => 'artists', 'order'=>['id','desc'], 'join' => [['artist_genres','artists.artist_genre_id','=','artist_genres.id']] ]);
        return DataTables::of($artists)
            ->editColumn('checkbox',function($artists){
                return '<div class="inline custom-checkbox"><input name="checked" id="checkboxAll'.$artists->id.'" type="checkbox" class="custom-control-input CheckBoxes" value="'.$artists->id.'"><label for="checkboxAll'.$artists->id.'" class="custom-control-label"></label></div>';
            }) 
            ->editColumn('image', function($artists){
                if($artists->image != '' && file_exists(public_path('/images/artist/'.$artists->image)))
                    $src = asset('/images/artist/'.$artists->image);
                else
                    $src = asset('images/sites/500x500.png');
                return '<img src="'.$src.'" alt="" class="img-fluid" width="60px" height="60px">';
            })
            ->editColumn('dob', function($artists){
                return date('Y-m-d', strtotime($artists->dob));
            })
            ->editColumn('created_at', function($artists){
                return Carbon::parse($artists->created_at)->diffForHumans(Carbon::now());
            })
            ->editColumn('status', function($artists){
                return '<div class="custom-switch"><input id="switch'.$artists->id.'" name="status" class="custom-control-input changeStatus" '.($artists->status == 1 ? 'checked':'').' type="checkbox" data-url="'.url('artist/status/'.$artists->id).'"><label class="custom-control-label" for="switch'.$artists->id.'"></label></div>';
            })
            ->addColumn('action', function ($artists) {
                return '<div class="button-list"><a class="btn btn-sm btn-success-rgba" href="'.url('artist/edit/'.$artists->id).'"><i class="feather icon-edit-2"></i></a><button type="button" data-url="'.url('artist/destroy/'.$artists->id).'" class="btn btn-sm btn-danger-rgba" id="deleteUser"><i class="feather icon-trash"></i></button></div>';
            })
            ->rawColumns(['checkbox','image','status','action'])->make(true);
    }

    public function addEditArtist(Request $request, $id){
       
        $rules = [ 
            'artist_name' => 'required',
            'artist_genre' => 'required',
            'image' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ];
      
        $checkValidate = validation($request->except('_token'), $rules);
        if($checkValidate['status'] == 1){
            $slug = Str::slug($request->artist_name,'-');
            $where = is_numeric($id) ? [['id','!=',$id],['artist_name','=',$slug]] : [['artist_name','=',$slug]];
            $checkArtist = Artist::where($where)->first();
            if(!empty($checkArtist)){
                $resp = array('status'=>0, 'msg'=>__('adminWords.artist').' '.__('adminWords.already_exist'));
            }else{
                $checkArtist = is_numeric($id) ? Artist::find($id) : [];
                $data = $request->except('_token');
                $data['artist_genre_id'] = $request->artist_genre;
                $data['artist_slug'] = $slug;
                $data['status'] = isset($request->status) ? 1 : 0;
                $data['is_featured'] = isset($request->is_featured) ? 1 : 0;
                $data['is_trending'] = isset($request->is_trending) ? 1 : 0;
                $data['is_recommended'] = isset($request->is_recommended) ? 1 : 0;
                unset($data['artist_genre']);
                // ========================================
                if ($file = $request->file('image')) {
                    $validator = Validator::make(
                        [
                            'image' => strip_tags($request->image),
                            'extension' => strtolower($request->image->getClientOriginalExtension()),
                        ],
                        [
                            'image' => 'required',
                            'extension' => 'required|in:jpg,png',
                        ]
                    );
                    if ($validator->fails()) {
                        return back()->withErrors( __('Invalid file !'));
                    }
                    if ($file = $request->file('image')) {
                        $name = time() . $file->getClientOriginalName();
                        $file->move('images/artist', $name);
                        $data['image'] = $name;
                    }
                }
                // ========================================
                $addArtist = empty($checkArtist) ? Artist::create($data) : $checkArtist->update($data);
                $resp = ($addArtist) ? ['status'=>1, 'msg'=>__('adminWords.artist').' '.__('adminWords.success_msg')] : ['status'=>0, 'msg'=>__('adminWords.delete_success')];
            }
        }else{
           $resp = $checkValidate;
        }
        echo json_encode($resp);
    }
    
    public function createArtist(){
        $data['artistGenre'] = ArtistGenre::pluck('genre_name','id')->all();
        return view('artist::artist.addEdit', $data);
    }

    public function editArtist($id){
        
        $artistData = Artist::where('id',$id)->get();
        $data['artistData'] = !empty($artistData) ? $artistData[0] : [];
        $data['artistGenre'] = ArtistGenre::pluck('genre_name','id')->all();
        return view('artist::artist.addEdit', $data);
    }

    function updateArtistStatus(Request $request, $id){
        $checkValidate = validation($request->all(),['status' =>'required']);
        if($checkValidate['status'] == 1){
            $resp = change_status(['table'=>'artists', 'column'=>'id', 'where'=>['id'=>$id],'data'=> ['status'=>$request->status]]);
            echo $resp;
        }else{
            echo json_encode($checkValidate);
        }
    }

    public function destroyArtist($id){
        $resp = singleDelete([ 'table'=>'artists','column'=>['image','artist_name'], 'where'=>['id'=>$id], 'msg'=> __('adminWords.artist').' '.__('adminWords.delete_success'), 'isImage'=>public_path().'/images/artist/' ]);
        echo $resp;        
    }

    function bulkDeleteArtistData(Request $request){
        $checkValidate = validation($request->all(),['checked' =>'required'], __('adminWords.atleast').' '.__('adminWords.artist').' '.__('adminWords.must_selected') );
        if($checkValidate['status'] == 1){
            $resp = bulkDeleteData(['table'=>'artists','column'=>'id', 'msg'=>__('adminWords.artist').' '.__('adminWords.delete_success'), 'request'=>$request->except('_token')]);
        }else{
            $resp = $checkValidate;
        }
        echo json_encode($resp);
    }

    public function artistGenres(){
        return view('artist::artist_genre.index');
    }

    public function artistGenreData(){
        $artist_genre = select(['table'=>'artist_genres','column'=>'*','order'=>['id','desc']]);
        return DataTables::of($artist_genre)
        ->editColumn('checkbox',function($artist_genre){
            return '<div class="inline custom-checkbox"><input name="checked" id="checkboxAll'.$artist_genre->id.'" type="checkbox" class="custom-control-input CheckBoxes" value="'.$artist_genre->id.'"><label for="checkboxAll'.$artist_genre->id.'" class="custom-control-label"></label></div>';
        })
        ->editColumn('created_at', function($artist_genre){
            return Carbon::parse($artist_genre->created_at)->diffForHumans(Carbon::now());
        })
        ->addColumn('action', function ($artist_genre){
            return '<div class="button-list"><a class="btn btn-sm btn-success-rgba artistGenrePopup" data-url="'.url('genre/data/'.$artist_genre->id).'" data-save="'.url('artist/genre/addEdit/'.$artist_genre->id).'"><i class="feather icon-edit-2"></i></a><button type="button" data-url="'.url('artist/genre/destroy/'.$artist_genre->id).'" class="btn btn-sm btn-danger-rgba" id="deleteUser"><i class="feather icon-trash"></i></button></div>';
        })
        ->rawColumns(['checkbox','status','action'])->make(true);
    }

    public function getArtistGenreData($id){
        $genre = ArtistGenre::find($id);
        if(!empty($genre)){
            $resp = ['status'=>1, 'data'=>$genre];
        }else{
            $resp = ['status'=>0, 'msg'=>__('adminWords.error_msg')];
        }
        echo json_encode($resp);
    }

    public function updateArtistGenre(Request $request, $id){ 
        $checkValidate = validation($request->all(),['status' =>'required']);
        if($checkValidate['status'] == 1){
            $resp = change_status(['table'=>'artist_genres', 'column'=>'id', 'where'=>['id'=>$id],'data'=> ['status'=>$request->status]]);
            echo $resp;
        }else{
            echo json_encode($checkValidate);
        }
    }

    public function addEditArtistGenre(Request $request, $id){
        $checkValidate = validation($request->except('_token'), ['genre_name' => 'required'] );
        if($checkValidate['status'] == 1){
            $arr = [
                'genre_name' => $request->genre_name,
                'genre_slug' => Str::slug($request->genre_name,'-'),
            ];
            $where = is_numeric($id) ? [['id','!=',$id], ['genre_slug','=', $arr['genre_slug']] ] : [['genre_slug','=', $arr['genre_slug']]];
            $artistGenre = ArtistGenre::where($where)->get();
            if(count($artistGenre) > 0){
                $resp = ['status'=>0, 'msg'=> __('adminWords.genre').' '.__('adminWords.already_exist')];
            }else{
                $genre = is_numeric($id) ? ArtistGenre::find($id) : [];
                if(!empty($genre)){
                    $genre->update($arr);
                    $msg = __('adminWords.genre').' '.__('adminWords.updated_msg');
                }else{
                    ArtistGenre::create($arr);
                    $msg = __('adminWords.genre').' '.__('adminWords.added_msg');
                }
                $resp = ['status'=>1, 'msg'=>$msg];
            }
        }else{
            $resp = $checkValidate;
        }
        echo json_encode($resp);
    }

    public function destroyArtistGenre($id){
        $resp = singleDelete([ 'table'=>'artist_genres','column'=>'id','where'=>['id'=>$id], 'msg'=>__('adminWords.genre').' '.__('adminWords.delete_success') ]);
        echo $resp;
    }

    public function bulkDeleteArtistGenre(Request $request){
        $checkValidate = validation($request->all(),['checked' =>'required'],__('adminWords.atleast').' '.__('adminWords.genre').' '.__('adminWords.must_selected') );
        if($checkValidate['status'] == 1){
            $resp = bulkDeleteData(['table'=>'artist_genres', 'column'=>'id', 'msg'=>__('adminWords.genre').' '.__('adminWords.delete_success'),'request'=>$request->except('_token')]);
        }else{
            $resp = $checkValidate;
        }
        echo json_encode($resp);
    }

}
